/****************************************************************************************************************************************************
* 
*	Author: GerryL (Backshed)
* 
*	History:
*	Version 1.0 : June 20. Initial developement. 
*	Version 1.1 : 10 March 21
*		1.	Fixed issue of missing bytes between functions that sometimes occurs.
*	    2.  Fixed issue with extracting calling format.
*		3.  Fixed issue with resolving static data.
* 
*	 Licence:
*   This code is covered by the licence as detailed at https://unlicense.org/ , which is essentially its public domain, but "AS IS", WITHOUT WARRANTY OF ANY KIND.
* 
* 		Overview:
*	Will convert an ELF file that is generated for the MMPlus (PIC32MX470) or CMM2 (STM32H743IIT6) into the correct
*	format of a CSUB or CFunction (MMPlus only, CMM2 does not support a CFunction) so as to be called by MMBasic on these platforms. 
*	The generated CSub/CFunction is saved to a .bas text file and also placed on the clip board.
*	This code has been sucessfully compiled/linked on Visual Studio 2019 Professional.
* 
*		Usage:
*	This class has been used in a simple windows console application and also in more complex GUI Window applications.  
*   Refer to the code in the windows console applicatin ConCFuncSubGen.cpp as a guide.
*	To create a MMPlus or CMM2 Cfunction or CSub text from relevant ELF file the following calls are made:
*	1.	Create the class as,
*			std::unique_ptr<CElfToCSub> pElf(const std::wstring& ELFFile, const std::wstring& Mergename, const std::wstring& CProgFile, 
*			const int CMergeJoinType = MERGE_TYPE, const int CFuncSubType = CSUB_TYPE);
*			Note: CProgFile (the C source file path) can be an empty string which will prevent adding the calling convention.
*			if CProgFile is set to a C source file path then the calling convention comment added to the CSub/CFunction is 
*			derived from the C source and then added in the CSub/CFunction text as a comment to assist users. 
*			Note for extremely large CSUB's (i.e 20K+) generating the calling convention  may take a few seconds.  For smaller CSub sizes the time taken is small.
*	2.	Load the ELF file as,
*			int nBoardType = pElf->Load_ELFFile(); // nBoardType will be CMM2_TYPE or MMPLUS_TYPE or an Error (less than 0)
*	3.  Create the CFunction or CSub text in a Basic file and on clipboard as,
*			pElf->CreateCFunctions();
*
*		Notes:
*	1.	MMPlus compiler Optimisation. 
*		Optimisation of -O1 must be set.
*	2.	CMM2 Optimisation
*		Optimisation other than -O0 can cause unintended issues (and sometimes fix unintended issues) and as such caution should be used in changing optimisation.
*		Correctly loading constant data ( static const char mystring[]="This is a string") as per Reference 2 with optimisation other than -O0 can 
*		sometimes be problematic.
* 	3.  MMPlus generated ELF's tends to place static constant data in the .rodata area while CMM2 generated ELF's tends to use the .text area and flag it as
*		an object type as apposed to a function type. 	  
*	4.	For MMPlus if a JAL op code ('0F4', jump & link) in encountered it is changed to an unconditional branch with op code '041' with
*		a relative jump calculated (how many 32 bit words to jump relative from current position to the original jump location).
*   5.  The CMM2 may mix 32 bit and 16 bit machine commands which is not an issue for MERGE, this program will correctly define the start and end of JOIN functions.
*
*		Reference:
*	The following document was used as a reference for the CElfToCSub class,
*	1.	Executable and Linkable Format (ELF) ( Tool Interface Standards (TIS) Portable Formats Specification, Version 1.1)
*		I found it at http://flint.cs.yale.edu/cs422/doc/ELF_Format.pdf.
* 
*	2.	Backshed post by Nathan (10 Jan 2016) on handling static constant data http://www.thebackshed.com/forum/ViewTopic.php?TID=8244
* 
*   3.  Backshed post by Matherp on CMM2 CFunctions,  https://www.thebackshed.com/forum/ViewTopic.php?TID=12289&PID=149258#149258
* 
*	4.	CircuitGizmos "Understanding CFunctions" (excerts of Matherp Backshed posts) at https://circuitgizmos.com/projects/microboard-micromite-projects/mmbasic-code-library/
*
**********************************************************************************************************************************************************
*/



#include "pch.h"
#include <time.h>
#include <iostream>  
#include <regex>
#include <string>
#include <stdlib.h>
#include <algorithm> 
#include "CElfToCSub.h"



CElfToCSub::CElfToCSub(const std::wstring& ELFFile, const std::wstring& Mergename, const std::wstring& CProgFile,
	const int CMergeJoinType, const int CFuncSubType) :m_wsELFFilePath{ ELFFile }, m_wsMergeName{ Mergename }, m_wsCFilePath{ CProgFile },
	m_MergeJoinType{ CMergeJoinType }, m_nSubFuncType{ CFuncSubType }
{
	/* Objective: Constructor
   * Input: 
   *	ELFFile, path of the ELF file to convert to a CFunction
   *	Mergename, if creating a Merge type of function the name to call the function, not used in Join function
   *	CProgFile, path of the C source file, can be empty string if there is no requirment to create a comment with calling convention
   *	CMergeJoinType, set to either MERGE_TYPE or JOIN_TYPE depending on requirment
   *	CFuncSubType, set to either CSUB_TYPE or CFUNC_TYPE depending on whether a CSUB or CFunction required.  Note that the type can be also set
   *    by calling the member SetFuncSubType after the CElfToCSub is constructed.
   */

	
}

CElfToCSub::~CElfToCSub()
{
	// tidy up
	m_SectionVector.clear(); 
	m_SymbolTableVector.clear(); 
	m_FunctionDataVector.clear();
}

void CElfToCSub::CopyToClipboard(std::wstring ws)
{
	// Objective: copy a wide string to the clipboard
   // Input: string to copy
	// Output: NIL.
	
	if (OpenClipboard(NULL))
	{
		EmptyClipboard(); // free existing global
		HGLOBAL hgClipBuffer = nullptr;
		std::size_t nBytes = (sizeof(wchar_t)* (ws.length()+1));
		hgClipBuffer = GlobalAlloc(GMEM_MOVEABLE, nBytes);
		if (!hgClipBuffer)
		{
			CloseClipboard();
			return;
		}
		wchar_t* gClipBoardBuffer = static_cast<wchar_t*>(GlobalLock(hgClipBuffer));
		memcpy(GlobalLock(hgClipBuffer), ws.c_str(), nBytes);
		GlobalUnlock(hgClipBuffer);
		SetClipboardData(CF_UNICODETEXT, hgClipBuffer);  
		CloseClipboard();
	}
}

long CElfToCSub::CreateCFunctions()
{
	/* ==========================================================================================================================
	* Prerequisites: CElfToCSub::Load_ELFFile() must have been previously called.
	* Objective: 
	*	1.	Creates the actual CSub or CFunction code from the ELF file and places it in m_wstrCode. Called once ELF file loaded.
	*	2.	Saves the m_wstrCode on the Windows clipboard.
	*	3.	Saves the m_wstrCode in a BASIC (.bas) file with Title the same as the ELF Title. 
	*		Note if the CElfToCSub class is created with the C source file (m_wsCFilePath) set then the .bas file path is the same as the C source file 
	*		else its the same path as the ELF file. (in the Windows GUI usage the C file is set as it is required by the batch file to comiple and link). 
	*	4.  Creates the CSub or CFunction calling statement in m_wstrCode if C source file is set when class created. 
	*
	* Input: Nil
	* 
	*  Output: 
	*		If no errors encountered then the byte size of the generated MMBasic file that contains the CFunction or CSub (+ number).
	*		Errors (negative number):
	*		ERROR_ELF_OPEN, could not open elf file
	*		ERROR_ELF_STRING_TABLE_NOT_FOUND, string table was not loaded correctly in CElfToCSub::Load_ELFFile()
	*		ERROR_ELF_BASIC_FILE_CREATE, Could not create the BASIC (.bas) file.
	*		ERROR_TIME_STRUCT_FILL, could not fill tm structure with time
	*		ERROR_TIME_ASC_CONVERT, could not generate ascii time
	*		ERROR_ELF_C_FILE_OPEN, could not open C file
	*		ERROR_ELF_HDR_BYTES, File Header read bytes Incorrect
	*		ERROR_ELF_SECTION_HDR_SIZE, File Section Header Size Incorrect
	*		ERROR_ELF_SECTION_HDR_READ, Section Header bytes read incorrect
	*		ERROR_ELF_SYMB_TABLE_READ, Symbol Table bytes read incorrect
	*		ERROR_ELF_NOT_ELF_FILE, Not a correct ELF file as not 'ELF' word found
	*		ERROR_JOIN_CONST_DATA, found static con
	* 
	*       ========================================================================================================*/

	// get just the C routines or static const in .text data and ignore the rest
	// sort out the jump if a merge and collect some stats
	m_wsFunctionInfo.clear();
	m_wsFunctionInfo = L"	             *** Function Information ***\r\n";
	m_wsFunctionInfo += L"	                    NAME    ADDRESS      SIZE \r\n";
	UINT buffcnt{};
	wchar_t wbuff[80];
	m_ELFFile.open(m_wsELFFilePath, std::ifstream::in | std::ifstream::binary);
	if (m_ELFFile.good()) {
		for (std::vector<Elf32_Sym>::iterator it = m_SymbolTableVector.begin(); it != m_SymbolTableVector.end(); ++it) {
			if (it->st_shndx == m_nTextSectionIndex) { // this is the .text section
				// check that there is no CMM2 static const data in a JOIN type
				if (m_MergeJoinType == JOIN_TYPE && (it->st_info & 0xf) == STB_OBJECT)
					return ERROR_JOIN_CONST_DATA;
				// now load stats into m_wsFunctionInfo 
				char cbuff[255];
				if (m_nStrTabIndex != -1) { // has a string table been found, avoid indexing to nothing
					UINT offset = it->st_name + m_SectionVector[m_nStrTabIndex].sh_offset;
					m_ELFFile.seekg(offset, std::ifstream::beg); // at correct string in table
					buffcnt = 0;
					do { // load its C code name 
						m_ELFFile.read(&cbuff[buffcnt++], 1);
					} while (cbuff[buffcnt - 1] != '\0');
					char atexit[10];
					strcpy_s(atexit, "atexit"); // check for atexit
					int ncmp = strcmp(cbuff, atexit);
					if (cbuff[0] != '_' && ncmp != 0) { // only select user functions 
						std::string sname(cbuff);// here we have correct section and name 
						int nsize = MultiByteToWideChar(CP_UTF8, 0, &sname[0], (int)sname.size(), NULL, 0);// last param = 0 so requird size returned
						std::wstring wswidename(nsize, 0);// string of correct size
						MultiByteToWideChar(CP_UTF8, 0, &sname[0], (int)sname.size(), &wswidename[0], nsize);
						it->st_wsName = wswidename;
						// has data and exclude local (e.g recursive calls)
						if (it->st_size > 0 && it->st_info != LOCAL_AND_FUNCTION) {
							m_CRoutineSymbolTableVector.push_back(*it);
							swprintf_s(wbuff, 80, L"%32s    %08I32X     %08I32X\r\n", wswidename.c_str(), it->st_value, it->st_size);
							m_wsFunctionInfo += wbuff;
						}
						// sort out merge jump
						if (m_MergeJoinType == MERGE_TYPE) {
							if (wswidename == L"main") {
								m_nMainJmp = it->st_value & MAX_CODE_BYTES;
								m_nMainAddress = it->st_value & MAX_CODE_BYTES;
								if (m_nELFType == CMM2_TYPE) {
									m_nMainJmp -= 1;
									m_nMainAddress -= 1;
								}
								m_nMainJmp /= CFUNC_CMD_BYTES_LENGTH;
								if (m_nMainAddress != (m_nMainJmp * CFUNC_CMD_BYTES_LENGTH)) // if no remainder then equal
									m_bMainNot32bitBoundary = true;
								else
									m_bMainNot32bitBoundary = false;
							}
						}//merge type
					} // correct name
				}// found a string table
				else {
					m_ELFFile.close();
					return ERROR_ELF_STRING_TABLE_NOT_FOUND;
				}
			}// text
		} // for
	} // good open ELF file
	// now sort .text in order of address from low to hi
	std::sort(std::begin(m_CRoutineSymbolTableVector), std::end(m_CRoutineSymbolTableVector), [](const Elf32_Sym& a, const Elf32_Sym& b) {
		return a.st_value < b.st_value;
		});
	// check for static const data in .rodata in a MMPLUS JOIN type
	if (m_MergeJoinType == JOIN_TYPE && m_nRodataSize != 0) {
		return ERROR_JOIN_CONST_DATA;// warn user static data in JOIN is a no go
	}
	std::wstring csTempFileName;
	int nMainJumpLoc{};
	m_wstrCode.clear();
	m_wstrCode.reserve(10000);
	m_wstrCode =L"'File ";
	csTempFileName = m_wsELFFilePath;
	csTempFileName = csTempFileName.substr(csTempFileName.find_last_of(L'\\')+1, (csTempFileName.find_last_of('.') - csTempFileName.find_last_of(L'\\')-1));
	csTempFileName += BASIC_FILE_APPEND;
	m_wsBASICFileName = csTempFileName;
	m_nCCode4ByteCnt = 0;
	m_wstrCode +=  L" Written ";
	__time64_t rawtime;
	struct tm timeinfo;
	_time64(&rawtime);
	if(localtime_s(&timeinfo ,&rawtime))
		return ERROR_TIME_STRUCT_FILL;
	char buff2[30];
	if (asctime_s(buff2, 30, &timeinfo))
		return ERROR_TIME_ASC_CONVERT;
	std::string stime(buff2);
	int nsize = MultiByteToWideChar(CP_UTF8, 0, &stime[0], (int)stime.size(), NULL, 0);// last param = 0 so required size returned
	std::wstring wswide(nsize, 0);// string of correct size
	MultiByteToWideChar(CP_UTF8, 0, &stime[0], (int)stime.size(), &wswide[0], nsize);
	m_wstrCode += wswide; // stime;
	m_wstrCode += L"\r\n";
	m_nMainParamInsertIndex = m_wstrCode.length();// index where to insert calling format for Merge
	m_wstrCode += L"\r\n";
	if (m_MergeJoinType == MERGE_TYPE) {
		if(m_nELFType == CMM2_TYPE)
			m_wstrCode += L"CSUB "; // force to CSub regardless
		else if (m_nELFType == MMPLUS_TYPE) { // can be CSub or CFunction
			if(m_nSubFuncType == CFUNC_TYPE)
				m_wstrCode += L"CFUNCTION ";
			else if (m_nSubFuncType == CSUB_TYPE)
				m_wstrCode += L"CSUB ";
		}
		m_wstrCode += m_wsMergeName;
		m_wstrCode += L"\r\n";
	}
	
	if (m_ELFFile.good()) {
		if (m_MergeJoinType == MERGE_TYPE) {
			CreateMergeText();
		}
		else if (m_MergeJoinType == JOIN_TYPE) {
			for (std::vector<Elf32_Sym>::iterator it = m_CRoutineSymbolTableVector.begin(); it != m_CRoutineSymbolTableVector.end(); ++it) {

							// skip main in a JOIN type 
						if (it->st_wsName != L"main") {
							CreateJoinText(m_nELFType, it);// generates the actual CFunction/CSub from .text data
						}

			} // for it, end of .text print
		} // join type
		// now if needed pad with zeroes the size of .dinit if its located before .text and .rodata and past .text data so const data located correctly
		if (m_MergeJoinType != JOIN_TYPE && m_nRodataSize != 0) { // do not do const data on join and only do if we have rodata
			if (m_nDinitSize > 0 && m_nRodataAdd > m_nDinitAdd && m_nDinitAdd > m_nTextAdd) {
				int ncount = m_nDinitSize / CFUNC_CMD_BYTES_LENGTH;
				int nremainder = m_nDinitSize % CFUNC_CMD_BYTES_LENGTH;
				if (nremainder) ncount++;
				m_wstrCode += L"    ";
				for (int i = 0; i < ncount; i++) {
					m_wstrCode += L"00000000    ";
					if ((i % 8 == 0) && (i != 0)) {
						m_wstrCode += L"\r\n";
						m_wstrCode += L"    ";
					}
					else {
						m_wstrCode += L" ";
					}
				}
				m_wstrCode += L"\r\n";
			}
			// now add .rodata
			int c4byteCount{};
			unsigned int nReadByteCnt{};
			UINT32 lnData{};
			wchar_t wbuff[20];
			int noffset = m_SectionVector[m_nRodataSectionIndex].sh_offset;
			m_ELFFile.seekg(noffset, std::ifstream::beg);
			m_wstrCode += L"\r\n    ";
			// add the static const data to output string
			while (nReadByteCnt < m_nRodataSize) {
				m_ELFFile.read((char*)&lnData, CFUNC_CMD_BYTES_LENGTH);
				nReadByteCnt += (long)m_ELFFile.gcount();
				swprintf_s(wbuff, 20, L"%08I32X", lnData);
				m_wstrCode += wbuff;
				swprintf_s(wbuff, 20, L"%08I32X", lnData);
				c4byteCount++;
				if ((c4byteCount % 8 == 0) && (c4byteCount != 0)) {
					m_wstrCode += L"\r\n";
					m_wstrCode += L"    ";
				}
				else {
					m_wstrCode += L" ";
				}
			}
		} // in not join and .rodata present
		m_ELFFile.close();
		m_wstrCode += L"\r\n";
		if (m_MergeJoinType == MERGE_TYPE) {
			if (m_nELFType == CMM2_TYPE)
				m_wstrCode += L"END CSUB"; // force to End CSub 
			else if (m_nELFType == MMPLUS_TYPE) { // can be CSub or CFunction End
				if (m_nSubFuncType == CFUNC_TYPE)
					m_wstrCode += L"END CFUNCTION";
				else if (m_nSubFuncType == CSUB_TYPE)
					m_wstrCode += L"END CSUB";
			}
		}
		if(m_wsCFilePath != WS_NOT_SET && !m_wsCFilePath.empty()) // may be empty for console use
			SetCallFormat(); // only do if C file path has been set 
		CopyToClipboard(m_wstrCode);
		// now create a Basic file to save the CFunction/CSub
		std::wofstream cfile;
		if (m_wsCFilePath.empty() || m_wsCFilePath.find(WS_NOT_SET) != std::wstring::npos) // in windows GUI application generated CFunction/CSub BASIC file placed in C file folder else same as ELF path
			m_wsBASICFilePath = m_wsELFFilePath;
		else m_wsBASICFilePath = m_wsCFilePath;
		m_wsBASICFilePath = m_wsBASICFilePath.substr(0, m_wsBASICFilePath.length() - (m_wsBASICFilePath.length() - m_wsBASICFilePath.find_last_of('\\')) + 1);
		m_wsBASICFilePath += csTempFileName;
		cfile.open(m_wsBASICFilePath, std::ofstream::out | std::ofstream::trunc | std::ofstream::binary);
		if (cfile.good()) {
			cfile.write(m_wstrCode.c_str(), m_wstrCode.length());
			cfile.flush();
			cfile.seekp(0, std::ofstream::end);
			long nFileSize = (long)cfile.tellp();
			cfile.close();
			return nFileSize; // good exit
		}// good basic file open
		else {
			return ERROR_ELF_BASIC_FILE_CREATE; // did not create a basic file
		}
		
	} // ELF file open ok
	else
		return ERROR_ELF_OPEN; // could not open ELF file
	
}

std::wstring CElfToCSub::RemoveCodeComments(const std::wstring& csLineIn)
{
	/* ================================
	* Prerequisites:
	* Objective:
	*	1.	Part of generating CSUB or CFunction calling convention string.
	*	2.	Removes any comments from a line of C code.
	*
	* Input: A String (csLineIn) to remove the comments from.
	*
	*  Output: A string with no C comments.
	*
	*       =================================*/
	std::wstring csRet(csLineIn);
	static bool bincomment{ false };
	char ncount{ 0 };
	int nstartmultilinecomment;
	int nlinecomment = csRet.find(L"//");
	if (nlinecomment > 0) nlinecomment--;
	if (nlinecomment != std::wstring::npos)
		csRet = csRet.substr(0, nlinecomment);
	// removes all code in comments
	while (ncount++ < 20) { // prevent inadvertant hang
		if (bincomment)
			nstartmultilinecomment = 0;
		else {
			nstartmultilinecomment = csRet.find(L"/*");
			if (nstartmultilinecomment != std::wstring::npos)
				bincomment = true;
		}
		int nendmultilinecomment = csRet.find(L"*/");
		if (nendmultilinecomment != std::wstring::npos)
			bincomment = false;
		// comment part of line /*xxxx */
		if (nstartmultilinecomment != std::wstring::npos && nendmultilinecomment != std::wstring::npos) {
			csRet.erase(nstartmultilinecomment, (nendmultilinecomment - nstartmultilinecomment) + 2);
		}
		else { // not found end of comment */ 
			if (nstartmultilinecomment != std::wstring::npos) {
				if (nstartmultilinecomment == 0)
					csRet = L"";
				else {
					csRet = csRet.substr(0, nstartmultilinecomment - 1);
					csRet += L" ";
				}
			}
			if (nendmultilinecomment != std::wstring::npos) {
				csRet += csRet.substr(nendmultilinecomment + 1);
			}
			else break;
		}
	} // while
	return csRet;
}

bool CElfToCSub::VoidFuncVoid(const std::wstring& sline, const std::wstring& name)
{
	/* ================================
	* Prerequisites: NIL
	* Objective:
	 *	1.	Part of generating CSUB or CFunction calling convention string.
	*	2.	Asseses part of a line of C code to see if a C function has a void return and void parameters,
	*	e.g void main(void),  main() etc, unlikely scenario but possible
	*
	* Input: A String (sline) to assess.
	*
	*  Output: True if the C function statement returns void and has no parameters, else return false.
	*
	*       =================================*/
	std::wsmatch match;
	std::wstring wsregstring(L"^[[:space:]]*(void)?[[:space:]]*(");
	wsregstring += name;
	wsregstring += L")[[:space:]]*[(][[:space:]]*(void)?[[:space:]]*[)]";
	std::wregex rvoid_name_void(wsregstring);
	if (std::regex_search(sline, match, rvoid_name_void))    //if true its [void] func ([void]), 
		return true;
	else return false;
	
}

std::wstring CElfToCSub::ExtractVariableType(const std::wstring& svariable)
{
	// empty string if not resolved
	/* ================================
	* Prerequisites: NIL
	* Objective:
	*	1.	Part of generating CSUB or CFunction calling convention string.
	*	2.	Asseses part of a line of C code to see what type of MMBasic variable is refrred to 
	*	(long long ->INTEGER, double-> FLOAT, char -> STRING etc)
	*	3. If var [] found will add "()" to the type to indicate an array. Note if an array is written as a pointer
			(i.e var1 *) it will not be flagged as an array as its ambigous
	*
	* Input: A String (sline) to assess, e.g, part of a C line where whole line may be 'long long FuncName(char[] var1,long long* var2, double* var3);'
	*
	*  Output: A string with the MMBasic variable type, empty string if correct type not found
	* 
	*
	*       =================================*/
	std::wstring swRect;
	std::wsmatch match;
	std::wregex rlong(L"[[:space:]]*(long[[:space:]]*[*]|long[[:space:]]*[[]|[[:space:]]*long )"); 
	std::wregex rchar(L"[[:space:]]*(char[[:space:]]*[*]|char[[:space:]]*[[]|[[:space:]]*char )");
	std::wregex rdoubler(L"[[:space:]]*(double[[:space:]]*[*]|double[[:space:]]*[[]|[[:space:]]*double )");
	std::wregex rint(L"[[:space:]]*(int[[:space:]]*[*]|int[[:space:]]*[[]|[[:space:]]*int )");
	std::wregex rmmfloat(L"[[:space:]]*(MMFLOAT[[:space:]]*[*]|MMFLOAT[[:space:]]*[[]|[[:space:]]*MMFLOAT )");
	std::wregex rCfloat(L"[[:space:]]*(float[[:space:]]*[*]|float[[:space:]]*[[]|[[:space:]]*float )");
	std::wregex rArrBracket(L"\\[[^\\]]*\\]");
	if (std::regex_search(svariable, match, rlong))   
		swRect = L"INTEGER";
	else if (std::regex_search(svariable, match, rchar))
		swRect = L"STRING";
	else if (std::regex_search(svariable, match, rdoubler))
		swRect = L"FLOAT";
	else if (std::regex_search(svariable, match, rmmfloat))
		swRect = L"FLOAT";
	else if (std::regex_search(svariable, match, rCfloat))
		swRect = L"FLOAT";
	else if (std::regex_search(svariable, match, rint))
		swRect = L"INTEGER";
	if (swRect.find(L"STRING") != std::wstring::npos && svariable.find(L"*") != std::wstring::npos) { // dont add brackets to STRING unless array of strings
		if (std::regex_search(svariable, match, rArrBracket))
			swRect += L"()";// add array brackets for char* str[]
	}
	else if (swRect.find(L"STRING") == std::wstring::npos) { // not a string
		if (std::regex_search(svariable, match, rArrBracket))
			swRect += L"()";// add array for others if bracket found
	}
	return swRect;
}

std::wstring CElfToCSub::CreateParsedParamFormat(const std::wstring& csline, const std::wstring& name)
{
	/* ================================
	* Prerequisites:
	* Objective:
	*	1.	Part of generating CSUB or CFunction calling convention string.
	*	2.	Asseses the parameters parsed to a C function in the C source file and then creates a string listing the type of MMBasic types,
	*	e.g. if C source file has a string of 'CFunction(long long* var1, char* var 2,double var3[]);' then translates to "INTEGER, STRING, FLOAT()"
	*
	* Input: A String (sline) to assess, e.g,  may be 'long long FuncName(char[] var1,long long* var2, double* var3);'
	*
	*  Output: A string with the MMBasic variable types seperated by comma, e.g INTEGER, STRING, FLOAT()
	*
	*       =================================*/
	std::wstring csRet;
	int nname = csline.find(name);
	int nfirstbracket = csline.find('(', nname);
	int nclosingbracket = csline.find(')', nfirstbracket);
	if (csline.find(L"void ", nfirstbracket) != std::wstring::npos) // found void in brackets!
		return L""; // this is an internal C call that uses void as parsed parmeter so skip
	int nstart = nfirstbracket - 1;
	if (m_MergeJoinType == MERGE_TYPE)
		csRet = m_wsMergeName;
	else if (m_MergeJoinType == JOIN_TYPE)
		csRet = name;
	csRet += L" ";
	if (m_nSubFuncType == CFUNC_TYPE)
		csRet += L"( ";
	std::wstring cstemp(csline);
	cstemp = cstemp.substr(nfirstbracket + 1, (nclosingbracket - nfirstbracket) - 1); // chars between (..)
	wchar_t tbuff[255];
	cstemp.copy(tbuff, csline.length());
	tbuff[csline.length() + 1] = '\0';
	wchar_t commasep[] = L",";
	wchar_t* next_token = NULL;
	wchar_t* token = NULL;
	token = wcstok_s(tbuff, commasep, &next_token);
	while (token != NULL) {
		// C file is case sensitive so below works
		cstemp = token;
		csRet += ExtractVariableType(cstemp);
		token = wcstok_s(NULL, commasep, &next_token);
		if(token != NULL)
			csRet += L", ";
	}
	return csRet;
}

std::wstring CElfToCSub::GetFullParamLine(std::wifstream* inFile, std::wstring& wsline)
{
	/* ================================
	* Prerequisites: inFile is at correct location to read the next line
	* Objective:
	*	1.	Part of generating CSUB or CFunction calling convention string.
	*	2.	Asseses the single line to see if it is a complete parsed parameter list, i.e contains the closing bracket
	*		If no closing bracket will read next line of file and add that to the input wsline, loops until find a closing bracket
	*
	* Input: A std::wstring ws line
	*
	*  Output: A string containing a complete function dec, e.g main(long long* var 1, char[] var2, float var3)
	* 
	*
	*       =================================*/
	std::wstring swtempline(wsline);
	wchar_t wbuff[255];
	while (!inFile->eof()) {
		if (swtempline.find(L")") != std::wstring::npos)
			break;// found closing bracket so exit
		inFile->getline(wbuff, 250);// read next line
		swtempline = wbuff;
		swtempline = swtempline.substr(0, static_cast<size_t>(inFile->gcount()));
		swtempline = RemoveCodeComments(swtempline);
		wsline += swtempline;
	}
	return wsline;
}

int CElfToCSub::SetCallFormat()
{
	/* ================================
	* Prerequisites: Requires the C source file path to have been set.
	* Objective:
	*	1.	Generates CSUB or CFunction calling convention string and then inserts it in relevant place in m_wstrCode.
	*		e.g. "Call Format for CSUB is Test INTEGER, STRING, FLOAT()"
	*			or Call Formatfor CFUNCTION Test (INTEGER, STRING, FLOAT()) AS STRING
	* Input: NIL
	*
	*  Output: NIL
	*
	*       =================================*/
	std::wifstream CFile;
	std::wstring ParameterString;
	std::wstring csline;
	wchar_t wbuff[500];
	int nInserOffset{};
	std::wsmatch match;
	CFile.open(m_wsCFilePath, std::ifstream::in);
	if (!CFile.good()) {
		return ERROR_ELF_C_FILE_OPEN;
	}
	// file is opened
	if (m_MergeJoinType == MERGE_TYPE) {
		//just  need to sort out the parameters in main
		while (!CFile.eof()) {
			CFile.getline(wbuff, 500);
			if (CFile.fail())
				break;
			csline = wbuff;
			csline = csline.substr(0, static_cast<size_t>(CFile.gcount()));
			csline = RemoveCodeComments(csline);
			if (VoidFuncVoid(csline, L"main")) { //not sure this is a realistic scenario but will capture
				ParameterString = m_wsMergeName; // no params
				CFile.close();
				return ELF_FUNC_OK;
			}
			else{
			// sort out main parameter list
				std::wregex rmain_param(L"[[:space:]]*((main[[:space:]]*[(]))"); // .. main(.. 
				if (std::regex_search(csline, match, rmain_param)) { // this line contains the 'main' function 
					// account for main parameters list being over multiple lines
					csline = GetFullParamLine(&CFile, csline);
					if (m_nSubFuncType == CSUB_TYPE)
						ParameterString = L"' Call Format for CSUB is \"  ";
					else if (m_nSubFuncType == CFUNC_TYPE)
						ParameterString = L"' Call Format for CFUNCTION is \"  ";
					ParameterString += CreateParsedParamFormat(csline, L"main");
					if (m_nSubFuncType == CFUNC_TYPE) {// return in main
						// need to resolve the return value
						ParameterString += L" ) ";
						auto nmain = csline.find(L"main");
						if (nmain != std::wstring::npos) {// technically should not be needed
							std::wstring wstemp = csline.substr(0, nmain);
							ParameterString += L" AS ";
							ParameterString += ExtractVariableType(wstemp);
						}
					}
					ParameterString += L" \"";
					break; // dont need to check any further
				}
			}
		}// while
		m_wstrCode.insert(m_nMainParamInsertIndex, ParameterString);// done
	} // if merge
	else if (m_MergeJoinType == JOIN_TYPE) {
		std::wstring jname;
		for (UINT i = 0; i < m_FunctionDataVector.size(); i++) {
			jname = m_FunctionDataVector[i].m_csName;
			CFile.seekg(0, std::ifstream::beg);
			if (m_nSubFuncType == CSUB_TYPE)
				ParameterString = L"  ' Call Format for CSUB is \"  ";
			else if (m_nSubFuncType == CFUNC_TYPE)
				ParameterString = L"  ' Call Format for CFUNCTION is \"  ";
			while (!CFile.eof()){
				CFile.getline(wbuff, 250);
				csline = wbuff;
				csline = csline.substr(0, static_cast<size_t>(CFile.gcount()));
				csline = RemoveCodeComments(csline);
				if (jname != L"main") { // in join we dont do main
					if (VoidFuncVoid(csline, jname))  //not sure this is a realistic scenario but will capture
						ParameterString += m_wsMergeName; // no params
					else{
						auto nfuncname = csline.find(jname);
						if (nfuncname != std::wstring::npos) {
							csline = GetFullParamLine(&CFile, csline);
							ParameterString += CreateParsedParamFormat(csline, jname);
							if (!ParameterString.empty()) {
								if (m_nSubFuncType == CFUNC_TYPE) {// return in function
									ParameterString += L" ) ";
									std::wstring wstemp = csline.substr(0, nfuncname);
									ParameterString += L" AS ";
									ParameterString += ExtractVariableType(wstemp);
									ParameterString += L" \"";
								}
								m_wstrCode.insert(m_FunctionDataVector[i].m_CodeStringParamLoc + nInserOffset, ParameterString);
								nInserOffset += ParameterString.length();
								break;
							}
						}// found name
					} // else not a void func void
				} // not main
			}// while looking thru file
		}// for each CFunction name
	}  // if join
	CFile.close();
	return ELF_FUNC_OK;
}

void CElfToCSub::CreateMergeText()
{
	/* ================================
	* Prerequisites: ELF file is open
	* Objective:
	*	1.	Called in CElfToCSub::CreateCFunctions()
	*	2.	Generates the actual CSub or CFunction text  in m_wstrCode for a Merged C function from the .text area
		* Input:
	*		NIL
	*
	*  Output: NIL
	*
	*       =================================*/

	int nFileoffset;
	Elf32_Addr nFirstFuncAddress;
	nFileoffset = m_SectionVector[m_nTextSectionIndex].sh_offset;
	m_ELFFile.seekg(nFileoffset, std::ifstream::beg);// at .text start
	UINT32 lnData{};
	nFirstFuncAddress = m_CRoutineSymbolTableVector[0].st_value;
	UINT32 nByteCounter{};
	wchar_t wbuff[80];
	int C4ByteCodeCnt{};
	m_nCCode4ByteCnt = 0;
	// add jump to main
	swprintf_s(wbuff, 80, L"    %08I32X\r\n", m_nMainJmp);
	m_wstrCode += wbuff;
	m_wstrCode += L"    ";
	while (nByteCounter < m_nTextSize) {
		if (m_nMainAddress == nByteCounter) { // have mains address, would only do if main on 32 bit boundary
			m_wstrCode += L" \r\n     'main \r\n";
			m_wstrCode += L"    ";
			C4ByteCodeCnt = 0;
		}
		m_ELFFile.read((char*)&lnData, CFUNC_CMD_BYTES_LENGTH);
		nByteCounter += (long)m_ELFFile.gcount();
		C4ByteCodeCnt++;
		m_nCCode4ByteCnt++;
		if (m_nELFType == MMPLUS_TYPE) {
			// need to determine if we have JAL op code of '0F4' (jump & link), 
			// if so turn it into a uncond branch with op code '041'
			// need to ascertain how many 32 bit words to jump relative from current position to the original jump location
			// the below equates for both forward and back branches
			if ((lnData & 0xfff00000) == 0x0f400000) {
				UINT32 nbranch = ((0xffff - (m_nCCode4ByteCnt - (lnData & 0xfffff))) + 1) | 0x04110000;
				lnData = nbranch;
			}
		}
		swprintf_s(wbuff, 20, L"%08I32X", lnData);
		m_wstrCode += wbuff; // csTemp;

		if ((C4ByteCodeCnt % 8 == 0) && (C4ByteCodeCnt != 0)) {
			m_wstrCode += L"\r\n";
			m_wstrCode += L"    ";
		}
		else {
			m_wstrCode += L" ";
		}
		if (!m_ELFFile.good())
			break; // somehow we have got to end of file or there is a file problem before end of .text!!
	}// while looping thru .text
}

void CElfToCSub::CreateJoinText(const int ELFType,  const std::vector<Elf32_Sym>::iterator& it)
{
	/* ================================
	* Prerequisites: ELF file open
	* Objective:
	*	1.	Called in CElfToCSub::CreateCFunctions()
	*	2.	Generates the actual CSub or CFunction text  in m_wstrCode for each C function from the .text area 
		* Input: 
	*		ELFType, ELF Type (CMM2_TYPE or MMPLUS_TYPE).
	*		FuncName, the name of the function that was stored in the Symbol Table of the ELF file.
	*		it is the iterator of the relevant m_CRoutineSymbolTableVector, which stores the routines and possibly static const data
	*
	*  Output: NIL
	*
	*       =================================*/
	
	int noffset;
	int nshift{};
	int C4ByteCodeCnt{};
	int ncount;
	int nremainder;
	long nReadByteCnt{};
	UINT32 lnData{};
	wchar_t wbuff[20];
	//CodeParams debugparam;
	int nSymbolType = it->st_info & 0xf;
	ncount = it->st_size / CFUNC_CMD_BYTES_LENGTH;
	nremainder = it->st_size % CFUNC_CMD_BYTES_LENGTH;
	if (nremainder) {
		ncount++; // add extra 4 byte read as is over an exact 4 bytes, on 32 bit boundary, any additional data would be ignored
	}
	if(ELFType == MMPLUS_TYPE)
		noffset = m_SectionVector[m_nTextSectionIndex].sh_offset + it->st_value;
	else if (ELFType == CMM2_TYPE) { // && nSymbolType == STT_FUNC)
		if( (it->st_info & 0x0f) != STB_OBJECT) // dont do on const vars in .text
			noffset = m_SectionVector[m_nTextSectionIndex].sh_offset + it->st_value - 1; // sub one as starts 800..01
		else
			noffset = m_SectionVector[m_nTextSectionIndex].sh_offset + it->st_value;
	}
	noffset &= MAX_CODE_BYTES; // get rid of the 8... in the address
	m_ELFFile.seekg(noffset, std::ifstream::beg);
	if ((it->st_info & 0x0f) == STT_FUNC) { // dont create functions for variable objects
		if (m_nELFType == CMM2_TYPE)
			m_wstrCode += L"CSUB ";// force to CSub regardless
		else if (m_nELFType == MMPLUS_TYPE) { // can be CSub or CFunction
			if (m_nSubFuncType == CFUNC_TYPE)
				m_wstrCode += L"CFUNCTION ";
			else if (m_nSubFuncType == CSUB_TYPE)
				m_wstrCode += L"CSUB ";
		}
		m_wstrCode += L" ";
		m_wstrCode += it->st_wsName;
		m_wstrCode += L"   ";// L"\r\n    ";
		FuncData funcdata;
		funcdata.m_csName = it->st_wsName;
		funcdata.m_nSymbolTableIndex = std::distance(std::begin(m_CRoutineSymbolTableVector), it); // nSybTabIndex;
		funcdata.m_CodeStringParamLoc = m_wstrCode.length();
		m_FunctionDataVector.push_back(funcdata);
		m_wstrCode += L"\r\n    ";
		swprintf_s(wbuff, 20, L"%08I32X\r\n    ", lnData);// zero offset
		m_wstrCode += wbuff; // csTemp;
	}
	for (int i = 0; i < ncount; i++) {
		m_ELFFile.read((char*)&lnData, CFUNC_CMD_BYTES_LENGTH);
		nReadByteCnt += (long) m_ELFFile.gcount();
		C4ByteCodeCnt++;
		m_nCCode4ByteCnt++;
		if (m_nELFType == MMPLUS_TYPE) {
			// need to determine if we have JAL op code of '0F4' (jump & link), 
			// if so turn it into a uncond branch with op code '041'
			// need to ascertain how many 32 bit words to jump relative from current position to the original jump location
			// the below equates for both forward and back branches
			if ((lnData & 0xfff00000) == 0x0f400000) {
				UINT32 nbranch = ((0xffff - (m_nCCode4ByteCnt-(lnData & 0xfffff)))+1) | 0x04110000;
				lnData = nbranch;
			}
		}
		swprintf_s(wbuff, 20, L"%08I32X", lnData);
		m_wstrCode += wbuff; // csTemp;
		
		if ((C4ByteCodeCnt % 8 == 0) && (C4ByteCodeCnt != 0)) {
			m_wstrCode += L"\r\n";
			m_wstrCode += L"    ";
		}
		else {
			m_wstrCode += L" ";
		}
	} // for
	if ((it->st_info & 0x0f) != STB_OBJECT)
		m_wstrCode += L"\r\n";
		if (m_nELFType == CMM2_TYPE)
			m_wstrCode += L"END CSUB"; // force
		else if (m_nELFType == MMPLUS_TYPE) { // can be CSub or CFunction
			if (m_nSubFuncType == CFUNC_TYPE)
				m_wstrCode += L"END CFUNCTION";
			else if (m_nSubFuncType == CSUB_TYPE)
				m_wstrCode += L"END CSUB";
		}
		m_wstrCode += L"\r\n";
	
}

int CElfToCSub::Load_ELFFile()
{
	/* ================================
	* Prerequisites: m_wsELFFilePath to be set to correct ELF file required.
	* Objective:
	*	1.	Loads the ELF file and then populates the m_SectionVector and m_SymbolTableVector.
	*	2.	Sets m_nTextSectionIndex to the correct index in m_SectionVector for latter use by CreateCFunctions()
	*	3.  Sets .text, .rodata and .dinit size and address if present
	* Input: Nil
	*
	* Output: 
	*		If all good then returns the type of platform the ELF is (i.e MMPLUS_TYPE or CMM2_TYPE), which are positive ints.
	*		Error (negative int)
	*		ERROR_ELF_OPEN if could not open ELF file
	*		ELF_HEADER_BYTES_ERROR  File Header read bytes Incorrect
	*		ERROR_ELF_SECTION_HDR_SIZE  File Section Header Size Incorrect
	*		ERROR_ELF_SECTION_HDR_READ  Section Header bytes read incorrect
	*		ERROR_ELF_SYMB_TABLE_READ  Symbol Table bytes read incorrect
	*		ERROR_ELF_NOT_ELF_FILE  Not a correct ELF file as "ELF" not found in hdr
	*       =================================*/

	m_ELFFile.open(m_wsELFFilePath, std::ifstream::in | std::ifstream::binary);
	if (!m_ELFFile.good()) {
		return ERROR_ELF_OPEN;
	}
	else { // all good
		m_nRodataSize = 0;
		m_nDinitSize = 0;
		m_nBSSSize = 0;
		long nReadCnt;
		Elf32_Word nByteCount{};
		char cbuff[255];
		UINT buffcnt{};
		m_nStrTabIndex = -1;
		m_ELFFile.seekg(0);
		m_ELFFile.read((char*)&m_ELF_Header, ELF_32_HEADER_SIZE);
		nReadCnt = (long) m_ELFFile.gcount();
		if (nReadCnt != ELF_32_HEADER_SIZE) {
			m_ELFFile.close();
			return ERROR_ELF_HDR_BYTES; // exit
		}
		if (m_ELF_Header.e_ident[0] == 0x7f &&
			m_ELF_Header.e_ident[1] == 'E' &&
			m_ELF_Header.e_ident[2] == 'L' &&
			m_ELF_Header.e_ident[3] == 'F') {
				// a good ELF file
				if (m_ELF_Header.e_machine == 0x28) {
					m_wsELEProcessorType = L"CMM2";
					m_nELFType = CMM2_TYPE;
				}
				else if (m_ELF_Header.e_machine == 0x8) {
					m_wsELEProcessorType = L" MMPlus";
					m_nELFType = MMPLUS_TYPE;
				}
				if (m_ELF_Header.e_shentsize != ELF_32_SECTION_HEADER_SIZE) {
					m_ELFFile.close();
					return ERROR_ELF_SECTION_HDR_SIZE; // exit
				}
				// cont..
				m_SectionVector.resize(m_ELF_Header.e_shnum);
				m_ELFFile.seekg(m_ELF_Header.e_shoff, std::ifstream::beg);
				for (int i = 0; i < m_ELF_Header.e_shnum; i++) {
					m_ELFFile.read((char*)&m_SectionVector[i], ELF_32_SECTION_HEADER_SIZE);
					nReadCnt = (long) m_ELFFile.gcount();
					if (nReadCnt != ELF_32_SECTION_HEADER_SIZE) {
						m_ELFFile.close();
						return ERROR_ELF_SECTION_HDR_READ; // exit
					}
				} //  end read sections
				// have all the section headers
				for (int i = 0; i < m_ELF_Header.e_shnum; i++) { // eval all sections
					if (m_SectionVector[i].sh_type == SHT_STRTAB) {// look for a string table
							//debugtemp1 = m_SectionVector[i];
							UINT offset = m_SectionVector[i].sh_name + m_SectionVector[m_ELF_Header.e_shstrndx].sh_offset;
							m_ELFFile.seekg(offset, std::ifstream::beg);
							buffcnt = 0;
							do { // load its name 
								m_ELFFile.read((char*)&cbuff[buffcnt++],1);
							} while (cbuff[buffcnt - 1] != '\0');
							if (cbuff[0] == '.' && cbuff[1] == 's' && cbuff[2] == 't' && cbuff[3] == 'r'
								&& cbuff[4] == 't' && cbuff[5] == 'a' && cbuff[6] == 'b')
								m_nStrTabIndex = i;
						} // end string table
						if (m_SectionVector[i].sh_type == SHT_TYPE_PROGBITS && (m_SectionVector[i].sh_flags == TEXT_ATTRIBUTE) || (m_SectionVector[i].sh_flags == DINIT_ATTRIBUTE) ) {
							// found a .text, .rodata or .dinit section
							UINT offset = m_SectionVector[i].sh_name + m_SectionVector[m_ELF_Header.e_shstrndx].sh_offset;
							m_ELFFile.seekg(offset, std::ifstream::beg);
							buffcnt = 0;
							do { // load its name (.text or .plt
								m_ELFFile.read((char*)&cbuff[buffcnt++], 1);
							} while (cbuff[buffcnt - 1] != '\0');
							if (strcmp(cbuff, ".text") == 0) {
								m_nTextSectionIndex = i;
								m_nTextAdd = m_SectionVector[i].sh_addr;
								m_nTextSize = m_SectionVector[i].sh_size;
							}
							if (strcmp(cbuff, ".rodata") == 0) {
								m_nRodataSectionIndex = i;
								m_nRodataAdd = m_SectionVector[i].sh_addr;
								m_nRodataSize = m_SectionVector[i].sh_size;
							}
							if (strcmp(cbuff, ".dinit") == 0) {
								m_nDinitSectionIndex = i;
								m_nDinitAdd = m_SectionVector[i].sh_addr;
								m_nDinitSize = m_SectionVector[i].sh_size;
							}
						
						} // if .text/.rodata/.dinit section
					if (m_SectionVector[i].sh_type == SHT_TYPE_SYMTAB) { // found symbol table
							// check name
							UINT offset = m_SectionVector[i].sh_name + m_SectionVector[m_ELF_Header.e_shstrndx].sh_offset;
							m_ELFFile.seekg(offset, std::ifstream::beg);
							buffcnt = 0;
							do { // load its name , .symtab 
								m_ELFFile.read((char*)&cbuff[buffcnt++], 1);
							} while (cbuff[buffcnt - 1] != '\0');
							m_nSymbolTableEntries = m_SectionVector[i].sh_size / m_SectionVector[i].sh_entsize;
							m_SymbolTableVector.resize(m_nSymbolTableEntries);
							m_ELFFile.seekg(m_SectionVector[i].sh_offset, std::ifstream::beg);
							for (UINT k = 0; k < m_nSymbolTableEntries; k++) { // load all symbols
								m_ELFFile.read((char*)&m_SymbolTableVector[k], m_SectionVector[i].sh_entsize);
								nReadCnt = (long) m_ELFFile.gcount();
								if (nReadCnt != ELF_32_SYMBOL_TABLE_ENTRY_SIZE) {
									m_ELFFile.close();
									return ERROR_ELF_SYMB_TABLE_READ; // exit
								}
							} // for k
							// have loaded symbol table
					} // if found symbol table
					if (m_SectionVector[i].sh_type == SHT_TYPE_SHT_NOBITS && m_SectionVector[i].sh_flags == SHT_ATTRIB_BSS ) {
						// found a .bss section
						UINT offset = m_SectionVector[i].sh_name + m_SectionVector[m_ELF_Header.e_shstrndx].sh_offset;
						m_ELFFile.seekg(offset, std::ifstream::beg);
						buffcnt = 0;
						do { // load its name (.bsst
							m_ELFFile.read((char*)&cbuff[buffcnt++], 1);
						} while (cbuff[buffcnt - 1] != '\0');
						if (strcmp(cbuff, ".bss") == 0) {
							m_nBSSSize = m_SectionVector[i].sh_size;
						}
					} // end of .bss
				} // for all sections
		} // end of a ELF file, detected ELF
		else {
			return ERROR_ELF_NOT_ELF_FILE;
		}
		m_ELFFile.close();
	} // ELF file opened OK
	return m_nELFType;
	
}

std::wstring CElfToCSub::BSSSize()
{
	/* ================================
	* Prerequisites:
	* Objective:
	*	1.	returns the size of BSS (bytes) as a string
	* Input:
	*		NIL
	*
	*  Output: The .bss size as a string string
	*
	*       =================================*/
	wchar_t wbuff[100];
	swprintf_s(wbuff, 100, L"                      .bss Size:    0x%08I32X", m_nBSSSize);
	return wbuff;
}

std::wstring CElfToCSub::TextSize()
{
	/* ================================
	* Prerequisites:
	* Objective:
	*	1.	returns the size of .text section (bytes) as a string
	* Input:
	*		NIL
	*
	*  Output: The .text size as a string string
	*
	*       =================================*/
	wchar_t wbuff[100];
	swprintf_s(wbuff, 100, L"                    .text Size:    0x%08I32X (%d Bytes)", m_nTextSize, m_nTextSize);
	return wbuff;
}
